#!/usr/bin/perl
#
# ZED80 - Z80 Experimental Disassembler in Perl
# Version: 0.9.1
# Copyright (C) 2006-2008 Ian Chapman
#
# Based upon documentation 'Decoding Z80 Opcodes' written by Cristian Dinu
# http://www.z80.info/decoding.htm
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# 0.9.0
# Initial Release
#
# 0.9.1
# Fixed trailing NOPs, thanks to Charles Mullins
#

package z80disasm;
use Exporter ();
use strict;

our $VERSION	= "1.00";		# Versionsnummer
our @ISA	= qw( Exporter );
our @EXPORT	= qw( $buffer $ptr $address $offset $pass decode );	# std.mig zu exportierende Symbole
our @EXPORT_OK	= qw( $debugm );			# bei Bedarf zu exportierende Symbole

# Config options
our $debugm = 0;          # Show debugging output
# End config options

# Globals
our $prefixmode = 0;     # Instruction prefix mode
our $ptr = -1;           # File location pointer
our $address = 0;        # Instruction address
our $dbyte = 0;          # Displacement byte holder for DDCB/FDCB prefixes.

our $buffer = '';	# Speicher zum Disassemblieren
our %marks;		# Markentabelle
our %addresses;		# Tabelle der Anfangsadressen der Befehle
our $offset = 0;	# Anfangsadresse/Offset
our $pass = 0;		# Durchlauf. pass=1 -> keine Anzeige

# Lookup tables
# | is used as a delimeter to indicate substitution when in DD/FD prefix mode
my @tab_r   = ('b', 'c', 'd', 'e', '|h|', '|l|', '|(hl)|', 'a'); # 8 bit registers
my @tab_rp  = ('bc', 'de', '|hl|', 'sp'); # Register pairs featuring SP
my @tab_rp2 = ('bc', 'de', '|hl|', 'af'); # Register pairs featuring AF
my @tab_cc  = ('nz', 'z', 'nc', 'c', 'po', 'pe', 'p', 'm'); # Condition codes
my @tab_alu = ('add	a,', 'adc	a,', 'sub	', 'sbc	a,', 'and	', 'xor	', 'or	', 'cp	'); # Arithmetic/Logic ops
my @tab_rot = ('rlc', 'rrc', 'rl', 'rr', 'sla', 'sra', 'sll', 'srl'); # Rotation/Shift Ops
my @tab_im  = ('0', '0/1', '1', '2', '0', '0/1', '1', '2'); # Interrupt modes
my @tab_bli = ([], [], [], [],
               ['ldi', 'cpi', 'ini', 'outi'],
               ['ldd', 'cpd', 'ind', 'outd'],
               ['ldir', 'cpir', 'inir', 'otir'],
               ['lddr', 'cpdr', 'indr', 'otdr']); # Block instructions


# open(FH, "<", "$ARGV[0]") || die "$!";
# binmode(FH);
# my @finfo = stat(FH);
# 
# while ($address < $finfo[7])
# {
#     my $val = &read8();
#     if ($debugm == 1)
#     {
#         print "\nX Y Z  P Q  DEC  HEX\n";
#         printf("%d %d %d  %d %d  %3d  %3X\n", &findx($val), &findy($val), &findz($val), &findp($val), &findq($val), $val, $val);
#     }
#     &nextinst($val);
# }
# 
# close (FH);
# 
# exit 0;

sub decode
{
    my $byte = &read8();
    my $pbits = &findp($byte);
    my $qbits = &findq($byte);
    my $xbits = &findx($byte);
    my $ybits = &findy($byte);
    my $zbits = &findz($byte);

     if ($debugm == 1)
     {
         print "\nptr X Y Z  P Q  DEC  HEX\n";
         printf("%.4X %d %d %d  %d %d  %3d  %3X\n", $ptr, $xbits, $ybits, $zbits, $pbits, $qbits, $byte, $byte);
     }


    # If we are currently in DD prefixmode and the next byte is DD, ED
    # or FD then it's equivalent to NONI then continue as normal
    if (($prefixmode == 0xDD) && (($byte == 0xFD) || ($byte == 0xDD) || ($byte == 0xED))) {&dumpinst('nop        ; noni');}
    # Check next byte to see if it's DDCB prefixed. If so, store displacement byte because it comes
    # BEFORE the opcode for these instructions.
    elsif (($prefixmode == 0xDD) && ($byte == 0xCB))
    {
        $prefixmode = 0xDDCB;
        $dbyte = &reads8();
    }
    # If we are currently in FD prefixmode and the next byte is DD, ED
    # or FD then it's equivalent to NONI then continue as normal
    elsif (($prefixmode == 0xFD) && (($byte == 0xFD) || ($byte == 0xDD) || ($byte == 0xED)))
    {
        &dumpinst('nop        ; noni');
    }
    # Check next byte to see if it's FDCB prefixed. If so, store displacement byte because it comes
    # BEFORE the opcode for these instructions.
    elsif (($prefixmode == 0xFD) && ($byte == 0xCB))
    {
        $prefixmode = 0xFDCB;
        $dbyte = &reads8();
    }
    elsif ($prefixmode == 0xED)
    {
        # Invalid instruction - equiv to NONI, NOP
        if (($xbits == 0) || ($xbits == 3)) {&dumpinst('nop        ; noni');}
        elsif ($xbits == 1) {&edx1($byte);}
        elsif ($xbits == 2) {&edx2($byte);}
    }
    elsif ($prefixmode == 0xCB)
    {
        if ($xbits == 0) {&dumpinst(sprintf('%s	%s', $tab_rot[$ybits], $tab_r[$zbits]));}
        elsif ($xbits == 1) {&dumpinst(sprintf('bit	%d,%s', $ybits, $tab_r[$zbits]));}
        elsif ($xbits == 2) {&dumpinst(sprintf('res	%d,%s', $ybits, $tab_r[$zbits]));}
        elsif ($xbits == 3) {&dumpinst(sprintf('set	%d,%s', $ybits, $tab_r[$zbits]));}
    }
    elsif ($prefixmode == 0xDDCB) {&ddcb($byte);}
    elsif ($prefixmode == 0xFDCB) {&fdcb($byte);}
    else # Normal mode
    {
        if ($xbits == 0)
        {
               if ($zbits == 0) {&x0z0($byte);}
            elsif ($zbits == 1) {&x0z1($byte);}
            elsif ($zbits == 2) {&x0z2($byte);}
            elsif ($zbits == 3)
            {
                if ($qbits == 0) {&dumpinst(sprintf('inc	%s', $tab_rp[$pbits]));}
                elsif ($qbits == 1) {&dumpinst(sprintf('dec	%s', $tab_rp[$pbits]));}
            }
            elsif ($zbits == 4) {&dumpinst(sprintf('inc	%s', $tab_r[$ybits]));}
            elsif ($zbits == 5) {&dumpinst(sprintf('dec	%s', $tab_r[$ybits]));}
            elsif ($zbits == 6)
            {
                # Fix for instance in DD/FD prefix mode, that dumpinst can't handle
                if (($prefixmode == 0xDD) && ($ybits == 6)) {&dumpinst(sprintf('ld	(ix%+d),%.3Xh', &reads8(), &read8()));}
                elsif (($prefixmode == 0xFD) && ($ybits == 6)) {&dumpinst(sprintf('ld	(iy%+d),%.3Xh', &reads8(), &read8()));}
                else {&dumpinst(sprintf('ld	%s,%.3Xh', $tab_r[$ybits], &read8()));}
            }
            elsif ($zbits == 7) {&x0z7($byte);}
        }
        elsif ($xbits == 1)
        {
            if (($zbits == 6) && ($ybits == 6)) {&dumpinst('halt');}
            else {&dumpinst(sprintf('ld	%s,%s', $tab_r[$ybits], $tab_r[$zbits]));}
        }
        elsif ($xbits == 2)
        {
            &dumpinst("$tab_alu[$ybits]$tab_r[$zbits]");
        }
        elsif ($xbits == 3)
        {
            if ($zbits == 0) {&dumpinst(sprintf('ret	%s', $tab_cc[$ybits]));}
            elsif ($zbits == 1) {&x3z1($byte);}
            elsif ($zbits == 2) {&dumpinst(sprintf('jp	%s,%s', $tab_cc[$ybits], mark(&read16())));}
            elsif ($zbits == 3) {&x3z3($byte);}
            elsif ($zbits == 4) {&dumpinst(sprintf('call	%s,%s', $tab_cc[$ybits], mark(&read16())));}
            elsif ($zbits == 5) {&x3z5($byte);}
            elsif ($zbits == 6) {&dumpinst(sprintf('%s%.3Xh', $tab_alu[$ybits], &read8()));}
            elsif ($zbits == 7) {&dumpinst(sprintf('rst	%d', $ybits*8));}
        }
    }
}


sub x0z0
{
    my $ybits = &findy($_[0]);

    if ($ybits == 0) {&dumpinst('nop');}
    elsif ($ybits == 1) {&dumpinst('ex	af,af\'');}
    elsif ($ybits == 2)
    {
        my $v=&reads8();
        &dumpinst(sprintf('djnz	%s', mark($v+$ptr+1 + $offset)));
    }
    elsif ($ybits == 3)
    {
        my $v=&reads8();
        &dumpinst(sprintf('jr	%s', mark($v+$ptr+1 + $offset)));
    }
    elsif ($ybits > 3)
    {
        my $v=reads8();
        &dumpinst(sprintf('jr	%s,%s', $tab_cc[$ybits-4], mark($v+$ptr+1 + $offset)));
    }
}


sub x0z1
{
    my $qbits = &findq($_[0]);
    my $pbits = &findp($_[0]);

    if ($qbits == 0) {&dumpinst(sprintf('ld	%s,%s', $tab_rp[$pbits], mark(&read16(), 'D') ));}
    elsif ($qbits == 1) {&dumpinst(sprintf('add	|hl|,%s', $tab_rp[$pbits]));}
}


sub x0z2
{
    my $qbits = &findq($_[0]);
    my $pbits = &findp($_[0]);

    if ($qbits == 0)
    {
        if ($pbits == 0) {&dumpinst('ld	(bc),a');}
        elsif ($pbits == 1) {&dumpinst('ld	(de),a');}
        elsif ($pbits == 2) {&dumpinst(sprintf('ld	(%s),|hl|', mark(&read16(), 'D')));}
        elsif ($pbits == 3) {&dumpinst(sprintf('ld	(%s),a', mark(&read16(), 'D')));}
    }
    elsif ($qbits == 1)
    {
        if ($pbits == 0) {&dumpinst('ld	a,(bc)');}
        elsif ($pbits == 1) {&dumpinst('ld	a,(de)');}
        elsif ($pbits == 2) {&dumpinst(sprintf('ld	|hl|,(%s)', mark(&read16(),'D')));}
        elsif ($pbits == 3) {&dumpinst(sprintf('ld	a,(%s)', mark(&read16(),'D')));}
    }
}


sub x0z7
{
    my $ybits = &findy($_[0]);

    if ($ybits == 0) {&dumpinst('rlca');}
    elsif ($ybits == 1) {&dumpinst('rrca');}
    elsif ($ybits == 2) {&dumpinst('rla');}
    elsif ($ybits == 3) {&dumpinst('rra');}
    elsif ($ybits == 4) {&dumpinst('daa');}
    elsif ($ybits == 5) {&dumpinst('cpl');}
    elsif ($ybits == 6) {&dumpinst('scf');}
    elsif ($ybits == 7) {&dumpinst('ccf');}
}


sub x3z1
{
    my $qbits = &findq($_[0]);
    my $pbits = &findp($_[0]);

    if ($qbits == 0) {&dumpinst(sprintf('pop	%s', $tab_rp2[$pbits]));}
    elsif ($qbits == 1)
    {
           if ($pbits == 0) {&dumpinst('ret');}
        elsif ($pbits == 1) {&dumpinst('exx');}
        # Many references use JP (HL) rather than JP HL. A 'syntax bug' of
        # sorts but JP HL is the technically correct form.
        elsif ($pbits == 2) {&dumpinst('jp	(hl)');}
        elsif ($pbits == 3) {&dumpinst('ld	sp,|hl|');}
    }
}


sub x3z3
{
    my $ybits = &findy($_[0]);

    if ($ybits == 0) {&dumpinst(sprintf('jp	%s', &mark(&read16() )));}
    elsif ($ybits == 1) {$prefixmode = 0xCB;} # Switch to CB prefixmode
    elsif ($ybits == 2) {&dumpinst(sprintf('out	(%.3Xh),a', &read8()));}
    elsif ($ybits == 3) {&dumpinst(sprintf('in	a,(%.3Xh)', &read8()));}
    elsif ($ybits == 4) {&dumpinst('ex	(sp),hl');}
    elsif ($ybits == 5)
    {
        # This instruction is an exception to the FD/DD prefix rule, so don't
        # use || delimeters to indicate HL substitution
        &dumpinst('ex	de,hl');
    }
    elsif ($ybits == 6) {&dumpinst('di');}
    elsif ($ybits == 7) {&dumpinst('ei');}
}


sub x3z5
{
    my $pbits = &findp($_[0]);
    my $qbits = &findq($_[0]);

    if ($qbits == 0) {&dumpinst(sprintf('push	%s', $tab_rp2[$pbits]));}
    elsif ($qbits == 1)
    {
        if ($pbits == 0) {&dumpinst(sprintf('call	%s', mark(&read16())));}
        elsif ($pbits == 1) {$prefixmode = 0xDD;} # Switch to DD prefix mode
        elsif ($pbits == 2) {$prefixmode = 0xED;} # Switch to ED prefix mode
        elsif ($pbits == 3) {$prefixmode = 0xFD;} # Switch to FD prefix mode
    }
}


sub edx1
{
    my $zbits = &findz($_[0]);
    my $qbits = &findq($_[0]);
    my $pbits = &findp($_[0]);
    my $ybits = &findy($_[0]);

    if ($zbits == 0)
    {
        if ($ybits == 6) {&dumpinst('in	(c)');}  # Some docs say IN 0,(C)
        else {&dumpinst(sprintf('in	%s,(c)', $tab_r[$ybits]));} # All other values are same instruction
    }
    elsif ($zbits == 1)
    {
        if ($ybits == 6) {&dumpinst('out	(c),0');}
        else {&dumpinst(sprintf('out	(c),%s', $tab_r[$ybits]));} # All other values are same instruction
    }
    elsif ($zbits == 2)
    {
        if ($qbits == 0) {&dumpinst(sprintf('sbc	|hl|,%s', $tab_rp[$pbits]));}
        elsif ($qbits == 1) {&dumpinst(sprintf('adc	|hl|,%s', $tab_rp[$pbits]));}
    }
    elsif ($zbits == 3)
    {
        if ($qbits == 0) {&dumpinst(sprintf('ld	(%s),%s', mark(&read16()), $tab_rp[$pbits]));}
        elsif ($qbits == 1) {&dumpinst(sprintf('ld	%s,(%s)', $tab_rp[$pbits], mark(&read16())));}
    }
    elsif ($zbits == 4) {&dumpinst('neg');}
    elsif ($zbits == 5)
    {
        if ($ybits == 1) {&dumpinst('reti');}
        else {&dumpinst('retn');} # All other values are same instruction
    }
    elsif ($zbits == 6) {&dumpinst(sprintf('im	%s', $tab_im[$ybits]));}
    elsif ($zbits == 7)
    {
        if ($ybits == 0) {&dumpinst('ld	i,a');}
        elsif  ($ybits == 1) {&dumpinst('ld	r,a');}
        elsif  ($ybits == 2) {&dumpinst('ld	a,i');}
        elsif  ($ybits == 3) {&dumpinst('ld	a,r');}
        elsif  ($ybits == 4) {&dumpinst('rrd');}
        elsif  ($ybits == 5) {&dumpinst('rld');}
        elsif  ($ybits == 6) {&dumpinst('nop');}
        elsif  ($ybits == 7) {&dumpinst('nop');}
    }
}


sub edx2
{
    my $zbits = &findz($_[0]);
    my $ybits = &findy($_[0]);

    if ($zbits < 4)
    {
        if ($ybits > 3) {&dumpinst($tab_bli[$ybits][$zbits]);}
        # Invalid instruction - equiv to NONI, NOP
        else {&dumpinst('nop       ; noni');}
    }
    # Invalid instruction - equiv to NONI, NOP
    else {&dumpinst('nop       ; noni');}
}


sub ddcb
{
    my $xbits = &findx($_[0]);
    my $ybits = &findy($_[0]);
    my $zbits = &findz($_[0]);

    if ($xbits == 0)
    {
        if ($zbits == 6) {&dumpinst(sprintf('%s	(IX%+d)',  $tab_rot[$ybits], $dbyte));}
        else {&dumpinst(sprintf('ld	%s,%s (IX%+d)', $tab_r[$zbits], $tab_rot[$ybits], $dbyte));}
    }
    elsif ($xbits == 1)
    {
        &dumpinst(sprintf('bit	%d,(IX%+d)', $ybits, $dbyte));
    }
    elsif ($xbits == 2)
    {
        if ($zbits == 6) {&dumpinst(sprintf('res	%d,(IX%+d)',  $ybits, $dbyte));}
        else {&dumpinst(sprintf('ld	%s,res %d,(IX%+d)', $tab_r[$zbits], $ybits, $dbyte));}
    }
    elsif ($xbits == 3)
    {
        if ($zbits == 6) {&dumpinst(sprintf('set	%d,(IX%+d)',  $ybits, $dbyte));}
        else {&dumpinst(sprintf('ld	%s,set %d,(IX%+d)', $tab_r[$zbits], $ybits, $dbyte));}
    }
}


sub fdcb
{
    my $xbits = &findx($_[0]);
    my $ybits = &findy($_[0]);
    my $zbits = &findz($_[0]);

    if ($xbits == 0)
    {
        if ($zbits == 6) {&dumpinst(sprintf('%s (IY%+d)',  $tab_rot[$ybits], $dbyte));}
        else {&dumpinst(sprintf('ld	%s,%s (IY%+d)', $tab_r[$zbits], $tab_rot[$ybits], $dbyte));}
    }
    elsif ($xbits == 1)
    {
        &dumpinst(sprintf('bit	%d,(IY%+d)', $ybits, $dbyte));
    }
    elsif ($xbits == 2)
    {
        if ($zbits == 6) {&dumpinst(sprintf('res	%d,(IY%+d)',  $ybits, $dbyte));}
        else {&dumpinst(sprintf('ld	%s,res %d,(IY%+d)', $tab_r[$zbits], $ybits, $dbyte));}
    }
    elsif ($xbits == 3)
    {
        if ($zbits == 6) {&dumpinst(sprintf('set	%d,(IY%+d)',  $ybits, $dbyte));}
        else {&dumpinst(sprintf('ld	%s,set %d,(IY%+d)', $tab_r[$zbits], $ybits, $dbyte));}
    }
}


# Bit manipulation routines for locating opcode 'sub codes'
sub findx() {return (($_[0] & 192) >> 6);}
sub findy() {return (($_[0] & 56) >> 3);}
sub findz() {return ($_[0] & 7);}
sub findq() {return (($_[0] & 8) >> 3);}
sub findp() {return (($_[0] & 48) >> 4);}


# Read unsigned byte
sub read8
{
    $ptr++;
    return unpack('C', substr($buffer, $ptr, 1));
}


# Read signed byte
sub reads8
{
    $ptr++;
    return unpack('c', substr($buffer, $ptr, 1));
}


# Read unsigned word
sub read16
{
#    $ptr = $ptr + 2;
#    $_ = unpack('v', substr($buffer, $ptr, 2));
    
    my $addr = read8();
    $addr += read8() * 0x100;
    
    return $addr;
}

sub hex16 {
	my $addr = shift;
	return sprintf("0%.4X", $addr);
}

sub mark {
	my ($address, $typ) = @_;
	$typ ||= 'M';

	if ( exists($marks{$address}) ) {
		return $marks{$address};
	} elsif ($pass > 0) {
		    # Markentabelle fllen
	    	$marks{$address} = $typ . &hex16($address);
		return $marks{$address};
	} else {
		return hex16($address);
	}
}


# Output the instruction, including performing necessary register substitutions
# when in prefix modes DD/FD
sub dumpinst
{
    my $inst = $_[0];

    if ($prefixmode == 0xFD) # Handle FD prefixmode substitution here
    {
        # If the instruction contains (HL), no further substitution should be done!
        if ($inst =~ m/\|\(hl\)\|/)
        {
            my $val = &reads8();
            $inst =~ s/\|\(hl\)\|/\(iy\+$val\)/g;
        }
        else
        {
            $inst =~ s/\|hl\|/iy/g;
            $inst =~ s/\|h\|/iyh/g;
            $inst =~ s/\|l\|/iyl/g;
        }
    }
    elsif ($prefixmode == 0xDD) # Handle FD prefixmode substitution here
    {
        # If the instruction contains (HL), no further substitution should be done!
        if ($inst =~ m/\|\(hl\)\|/)
        {
            my $val = &read8();
            $inst =~ s/\|\(hl\)\|/\(ix\+$val\)/g;
        }
        else
        {
            $inst =~ s/\|hl\|/ix/g;
            $inst =~ s/\|h\|/ixh/g;
            $inst =~ s/\|l\|/ixl/g;
        }
    }

    $inst =~ s/\|//g;
	
	# Adresse als gltig merken
    	$addresses{$address} = 1;

    if ($pass == 0) {

#    	print hex16($address), ' ';
	
	    if (exists $marks{$address}) {
	    	$_ = $marks{$address}.':';
	    } else {
	    	$_ = ' ';
	    }
	    printf("%s\t\t%s\n", $_ ,$inst);
	    
	}
    $address = $offset + $ptr + 1;
    $prefixmode = 0;
}


1;
