#!/usr/bin/perl
# ^^^ Remember to set this to the location of the perl
# interpreter on your system
#
# PINGMON V1.6 by Ian Chapman
#
# Pingmon is a simple program which tests basic network connectivity.
# It sends out a stream of pings at a user specified rate

use Term::ReadKey;
use Fcntl;
use Net::Ping;

open(IN,"</dev/tty");
*OUT = *IN;

#########################################
# *** START OF CONFIGURATION SECTION ***#
#########################################

$PROTOCOL = "icmp";
# Protocol to use for pinging. UNLESS YOU HAVE GOOD REASON FOR DOING
# SO, IT IS RECOMMENDED YOU LEAVE IT AS "icmp"
# icmp - Use ICMP (ala standard ping)
# tcp - Use TCP to test connectivity. Requires echo (TCP) running on remote host
# udp - Use UDP to test connectivity. Requires echo (UDP) running on remote host

$DELAY = 1;
# Time delay in seconds between scanning individual hosts.

$SCANDELAY = 0;
# Time delay in seconds between scans. That is, once all of your machines
# have been scanned, wait a specified number of seconds before scanning
# them again.

$SOUNDFILE = "/usr/share/sounds/hostoffline.wav";
# Default sound file to play when a host is offline

$PLAYER = "/usr/bin/play";
# The location of the player to use for playing sounds

$SOUND = 1;
# Start with sound on or off
# 0 = OFF, 1 = ON

$COLOUR = 1;
# Start with a colour or black & white display
# 0 = Black & White, 1 = Colour

$LOGFILE = "/var/log/pingmon.log";
# The file to write log entries to. Sorry no syslog yet, but you could
# probably pipe the output to the syslog using a utility such as syslogger
# If you do NOT want to log anything then either make sure log.txt is empty or
# change this to $LOGFILE = "/dev/null";

$CONF_IGNORE = "/etc/pingmoncfg/ignore.cfg";
$CONF_HOSTS = "/etc/pingmoncfg/hosts.cfg";
$CONF_LOGHOSTS = "/etc/pingmoncfg/loghosts.cfg";

########################################
# *** END OF CONFIGURATION SECTION ****#
########################################

###########################################
#PERL PROGRAMMERS ONLY, BEYOND THIS POINT.#
###########################################

$SIG{TERM} = sub {&Quit};
$SIG{QUIT} = sub {&Quit};
$SIG{ABRT} = sub {&Quit};
$SIG{HUP} = sub {&loadconfig};

&getterminfo();                 #Initialise Terminal
&loadconfig();                  #Load config
&ansisetup();                   #Setup ANSI variables
print $CLEAR;                   #Clear Screen
&cursmove(0,0);                 #Home cursor
$tot=4;                         #Set start of host output
$bounce = ($termx-1);           #Initialise for bouncing bricks
$lastdown = "None";             #Initialise name of last downed host
$lasttime = " since dawn of time       ";   #Initialise time of last downed host
$totdown = 0;                   #Initialise total downed hosts
%currendown = 0;		#Initialise current down

if ($SOUND == 1){$soundstatus = "ON  ";}
else {$soundstatus = "OFF ";}
    #Set the sound status depending on user's startup choice

&updateface;                    #Update the interface
&cursmove(1,4);                 #Move to start of host output

foreach $machine(@hosts)
{$currentdown{$machine}="NO";}


while ()                        # Start processing machines (pinging, logging etc)
{
    $totdown=0;
    &updateface;

    foreach $machine(@hosts)
    {&pinghost($machine);}

    for ($r=0; $r<$SCANDELAY; $r++)
    {    
        &keypresscheck;  
        sleep $SCANDELAY;
    }
}

exit 0;

sub playsound()                 #Subroutine to play the sound
{
    my $sndfile = $_[0];
    if ($SOUND == 1) {system("$PLAYER $sndfile 1>/dev/null 2>/dev/null");}
    # NOTE: You will need to change this if your OS does not
    # have a /dev/null device. Eg Windows, AmigaOS etc
}

sub pinghost()                  #Routine to ping host and display results
{
    my $host = $_[0];

    &consolecheck();            # Check if we need to clean up the console
    &keypresscheck();           # Check for a keypress

    if ($ignore{$host} eq "YES")    #Check if we need to ingnore this host
    {
        &cursmove(1, $tot);
        printf $BRICK . "$YELLOW %-7.7s " . $BRICK . "$CYAN %-12.12s " . $BRICK . "$PINK %-19.19s " . $BRICK . "$YELLOW OK         \n" . $WHITE, "IGNORED" , $host, "-------------------";
    }
    else
    {
        &cursmove(1, $tot);
        printf $BRICK . "$WHITE %-7.7s " . $BRICK . "$CYAN %-12.12s " . $BRICK . "$PINK %-19.19s " . $BRICK . " " , "PINGED", $host, scalar(localtime);
        $pingpacket = Net::Ping->new(lc($PROTOCOL));

        if ($pingpacket->ping($host, 3))
        {
             print $GREEN . "OK       \n" . $WHITE;
             $currentdown{$host}="NO";
        }
        else {&hstdown($host);}

        $pingpacket->close();
    }
    for ($r=0; $r<$DELAY; $r++)
    {
        &keypresscheck;
        sleep 1;
    }
}

sub cursmove                # Routine to control cursor movement. Uses ANSI
{
    my $x = $_[0];
    my $y = $_[1];
    print "\e[" . $y . ";" . $x . "H";
}

sub cursmoveprint           # Rotine to print at a specific location
{
    my $x = $_[0];
    my $y = $_[1];
    my $str = $_[2];
    &cursmove($x, $y);
    print $str;
}

sub updateface              # Redraw the interface to display changes
{
    &getterminfo();
    $titlestring = "PINGMON! V1.6 [IC SEP/2003]";
    &cursmoveprint(1, 1, $BRICK x $termx);
    &cursmoveprint(1, 2, $BRICK . " " x (($termx /2)-10) . $YELLOW . $titlestring . " " x (($termx /2)-10) . $BRICK);
    &cursmoveprint(1, 3, $BRICK x $termx);
    &cursmoveprint(1, $termy-7, $BRICK x $termx);
    &cursmoveprint(10, $termy-7, $BGBLUE . $WHITE . "Status:");
    &cursmoveprint(2, $termy-6, $RESETCOL . $RED . "          Sound: " . $YELLOW . $soundstatus);
    &cursmoveprint(2, $termy-5, $RED . "   Downed Hosts: " . $YELLOW . $totdown . " [on current scan]    ");
    &cursmoveprint(2, $termy-4, $RED . " Last Host Down: " . $YELLOW . $lastdown . $lasttime . "          ");
    &cursmoveprint(1, $termy-3, $BRICK x $termx);
    &cursmoveprint(10, $termy-3, $BGBLUE . $WHITE . "Control Panel:");
    &cursmoveprint(1, $termy-2, $BRICK . " " x ($termx - 2) . $BRICK);
    &cursmoveprint(3, $termy-2, $BGRED . "S" . $RESETCOL . "ound  $BGRED" . "C" . $RESETCOL . "olour  $BGRED" . "R" . $RESETCOL . "efresh  " . $BGRED . "D" . $RESETCOL . "isplay Downed Hosts");
    &cursmoveprint(1, $termy-1, $BRICK . " " x ($termx - 2) . $BRICK);
    &cursmoveprint(3, $termy-1, $BGRED . "+" . $RESETCOL . "/$BGRED-$RESETCOL Inc/Dec Delay ($DELAY)   " . "Re" . $BGRED . "l" . $RESETCOL . "oad Config  $BGRED" . "Q" . $RESETCOL . "uit");
    &cursmoveprint(1, $termy, $BRICK x $termx);
    for ($l = 1 ; $l < $termy-1 ; $l++)
    {
        cursmoveprint(1, $l , $BRICK);
        cursmoveprint($termx, $l , $BRICK);
    }
}

sub consolecheck    # Check if we need to redraw the interface, or clean
{                   # up the host output section
    @testtermsize = GetTerminalSize();
    if ($testtermsize[0] == 0) {$testtermsize[0] = 64;} # Some operating systems may return
    if ($testtermsize[1] == 0) {$testtermsize[1] = 24;} # 0 so we need default values

    if ($testtermsize[0] != $termx) # Check terminal X size to see if its changed
    {                               # if it has, the terminal has been resized
        print "$CLEAR";             # we need to refresh it
        $tot=4;
        &updateface;
    }

    if ($testtermsize[1] != $termy) # Check terminal Y size to see if its changed
    {
        print "$CLEAR";
        $tot=4;
        &updateface;
    }

    $tot++;
    $bounce--;

    if ($bounce == 0) {$bounce = ($termx-1);}

    &cursmoveprint($bounce, 1, $LBRICK);
    &cursmoveprint((($bounce * -1) + $termx), 1, $RBRICK);
    &cursmoveprint($bounce, 3, $LBRICK);
    &cursmoveprint((($bounce * -1) + $termx), 3, $RBRICK);

    if ($tot > $termy-8)
    {
       	$tot = 4 ;
        for ($z = 4; $z < $termy-7 ; $z++)
        {
            &cursmoveprint(1, $z, $BRICK . " " x (($termx) -2) . $BRICK);
	    }
		&cursmove(1,4);
	}
}

sub hstdown         # Deals with a downed host and whether it should be logged
{
    my $host = $_[0];

    print $RED . "DOWN" . $WHITE;

    $totdown++;
    $lastdown = $host;
    $lasttime = " \@ " . scalar(localtime);

    if ($log{$host} eq "YES")   # Check to see if we should log the downed host
    {
        open(LOG, ">>$LOGFILE");
            print LOG "$host : " . scalar(localtime) . "\n";
        close(LOG);
        print " (L)\n";
    }
    else
    {
        print " (-)\n";
    }

    $currentdown{$host}= scalar(localtime) . "\n";
    &updateface;
    &playsound($SOUNDFILE);
}

sub keypresscheck()         # Checks for key presses
{
    ReadMode 4, IN;
    $in = \*IN;
    $key=ReadKey(0.1);

    if (lc($key) eq "s")    # If s is pressed, alter the sound status
    {
        if ($SOUND == 1)
        {
            $soundstatus = "OFF ";
            &updateface;
            $SOUND = 0;
        }
        elsif ($SOUND == 0)
        {
            $soundstatus = "ON  ";
            &updateface;
            $SOUND = 1;
        }
    }

    if (lc($key) eq "c")    # If c is pressed, changed to colour/B&W
    {
        if ($COLOUR == 1)
        {
            $COLOUR = 0;
            &ansisetup();
            &updateface;
        }
        else
        {
            $COLOUR = 1;
            &ansisetup();
            &updateface();
        }
    }

    if (lc($key) eq "r")    # If R is pressed, refresh terminal
    {
        print "$CLEAR";
        $tot=4;
        &updateface;
    }

    if (lc($key) eq "l")    # If L is pressed, reload config
    {
        &loadconfig();
    }

    if (lc($key) eq "q")    # If Q is pressed, QUIT
    {
        &Quit;
    }

    if ($key eq "=" or $key eq "+")
    {
        $DELAY++;
	if ($DELAY > 9){$DELAY = 9;}
	&cursmoveprint(3, $termy-1, $BGRED . "+" . $RESETCOL . "/$BGRED-$RESETCOL Inc/Dec Delay ($DELAY)");
    }

    if ($key eq "-" or $key eq "_")
    {
        $DELAY--;
        if ($DELAY < 0){$DELAY = 0;}
	&cursmoveprint(3, $termy-1, $BGRED . "+" . $RESETCOL . "/$BGRED-$RESETCOL Inc/Dec Delay ($DELAY)");
    }

    if (lc($key) eq "d")
    {
	print $CLEAR . "\n";
        print "HOSTS CURRENTLY DOWN\n";
        print "--------------------\n";
	foreach $entry (@hosts)
	{
	    if ($currentdown{$entry} ne "NO")
	    {
		print "$entry : $currentdown{$entry}";
	    }
	}
        print "<Press any key to continue>\n";

        ReadKey(0, $in);
        print $CLEAR;
	&updateface;
    }

}

sub ansisetup()     # Setup ANSI variables depending on colour or B&W
{
    if ($COLOUR == 0)
    {
        $CLEAR    = "\e[2J";
        $CLINE    = "\e[K";
        $WHITE    = "\e[1;37m";
        $RED      = $WHITE;
        $GREEN    = $WHITE;
        $YELLOW   = $WHITE;
        $BLUE     = $WHITE;
        $PINK     = $WHITE;
        $CYAN     = $WHITE;
        $RESETCOL = "\e[0m";
        $BGBLUE = "\e[47m";
        $BGRED = "\e[1;40m";
        $BGCYAN = "\e[1;40m";
        $BRICK = $BGBLUE . " " . $RESETCOL;
        $LBRICK = $BGCYAN . " " . $BRICK;
        $RBRICK = $BRICK . $BGCYAN . " " . $RESETCOL;
        print $CLEAR;
    }
    else
    {
        $CLEAR    = "\e[2J";
        $CLINE    = "\e[K";
        $RED      = "\e[1;31m";
        $GREEN    = "\e[1;32m";
        $YELLOW   = "\e[1;33m";
        $BLUE     = "\e[1;34m";
        $PINK     = "\e[1;35m";
        $CYAN     = "\e[1;36m";
        $WHITE    = "\e[1;37m";
        $RESETCOL = "\e[0m";
        $BGBLUE = "\e[44m";
        $BGRED = "\e[1;41m";
        $BGCYAN = "\e[1;46m";
        $BRICK = $BGBLUE . " " . $RESETCOL;
        $LBRICK = $BGCYAN . "<" . $BRICK;
        $RBRICK = $BRICK . $BGRED . ">" . $RESETCOL;
        print $CLEAR;
    }
}

sub getterminfo()       # Get the terminal dimemsions
{
    @termsize = GetTerminalSize();
    $termx = $termsize[0];
    $termy = $termsize[1];

    if ($termx == 0){$termx = 64;} # Some operating systems may return 0
    if ($termy == 0){$termy = 24;} # so we need to set default values
}

sub loadconfig() #Load all the configurations
{
    @hosts = @empty;
    # Add the host names or IP addresses of any machines you
    # wish to monitor. The text file can also accept ranges
    open(SERV, $CONF_HOSTS);
        @machinelist = <SERV>;
    close(SERV);
    chomp(@machinelist);

    foreach $entry (@machinelist)
    {
        $entry = lc($entry);
	    if ($entry =~ m/!/)
        {
            @sections = split /:/, $entry;
            for ($num = $sections[1]; $num < ($sections[2]+1); $num++)
            {
                $ent = $sections[0];
                $ent =~ s/!/$num/g;
                push @hosts, $ent;
            }
        }
        else
        {
            push @hosts, $entry;
        }
    }

    #Place any machines you don't wish to be monitored in
    #the ingore list, perhaps because they may be down or
    #offline for quite a while. Also accepts ranges.

    open(IGN, $CONF_IGNORE);
        @temparr = <IGN>;
    close(IGN);
    chomp(@temparr);
    %ignore = ();

    foreach $entry (@temparr)
    {
        $entry = lc($entry);
        if ($entry =~ m/!/)
        {
            @sections = split /:/, $entry;
            for($num = $sections[1]; $num < ($sections[2]+1); $num++)
            {
                $ent = $sections[0];
                $ent =~ s/!/$num/g;
                $ignore{$ent} = "YES";
            }
        }
        else
        {
            $ignore{$entry} = "YES";
        }
    }

    # Place any machines in this file where a log entry should be made
    # if they go down

    open(LOG, $CONF_LOGHOSTS);
        @logdown = <LOG>;
    close(LOG);
    chomp(@logdown);
    %log = ();

    foreach $entry (@logdown)
    {
        $entry = lc($entry);
        if ($entry =~ m/!/)
        {
            @sections = split /:/, $entry;
            for($num = $sections[1]; $num < ($sections[2]+1); $num++)
            {
                $ent = $sections[0];
                $ent =~ s/!/$num/g;
                $log{$ent} = "YES";
            }
        }
        else
        {
            $log{$entry} = "YES";
        }
    }

        print $BGRED;
        print $CLEAR . "\n";
        sleep 1;
        print "\e[1;40m";
        print $CLEAR . "\n";
        &updateface();
}

sub Quit
{
    ReadMode 0, IN;
    close(IN);
    print $RESETCOL . $CLEAR;
    exit 0;
}
